/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2005 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_ether.cpp,v 1.37 2006/12/09 01:18:24 loic Exp $";
extern "C" {
#include "mx_arch.h"
#include "mx_misc.h"
#include "mx_instance.h"
#include "mx_malloc.h"
#include "mx_pio.h"
#include "mx_peer.h"
#include "mx_ether_common.h"
#include "mx_stbar.h"
#include "mx_pio.h"
}

#define __MBUF_TRANSITION_
#include <IOKit/network/IOEthernetController.h>
#include <IOKit/network/IONetworkController.h>
#include <IOKit/network/IOEthernetInterface.h>
#include <IOKit/network/IONetworkInterface.h>
#include <IOKit/network/IOMbufMemoryCursor.h>
#include <IOKit/network/IOBasicOutputQueue.h>

#include <IOKit/pci/IOPCIDevice.h>
#include <IOKit/IODeviceMemory.h>
#include <IOKit/IOInterruptEventSource.h>
#include <IOKit/IOTimerEventSource.h> 

#include "mx_driver.h"

#define __predict_true(exp)     __builtin_expect((exp), 1)
#define __predict_false(exp)    __builtin_expect((exp), 0)

#if MX_DARWIN_XX > 7
#define BIG_BYTES (mbuf_info.bigmclbytes)
#define SMALL_BYTES (mbuf_info.mhlen)
static	struct mbuf_stat mbuf_info;
#else
#define BIG_BYTES MCLBYTES
#define SMALL_BYTES MHLEN
#endif

class mx_IOEthernetInterface: public IOEthernetInterface
{
public:
	void set_mtu(IONetworkController *ctlr, int mtu);
};

void
mx_IOEthernetInterface::set_mtu(IONetworkController *ctlr, int mtu)
{
	ctlr->setMaxPacketSize(mtu + ETHER_HDR_LEN + ETHER_CRC_LEN);
	setMaxTransferUnit(mtu);
}

class mx_ethernet : public IOEthernetController
{
	OSDeclareDefaultStructors(mx_ethernet);
public:
	/* main driver's class */
	mx_driver *driver;
	mx_instance_state_t *is;
	struct mx_ether *eth;

	mx_IOEthernetInterface *netif;
	IOWorkLoop *mx_workloop;

	IONetworkStats  *net_stats;
	IOEthernetStats *ether_stats;
	IOOutputQueue *tx_queue;

	IOMbufBigMemoryCursor *tx_mcursor;
	IOMbufBigMemoryCursor *rx_small_mcursor;
	IOMbufBigMemoryCursor *rx_big_mcursor;
	
	bool enabled;
	bool stalled;

	OSDictionary *medium_dict;

        /*
	 *IOService (or its superclass) methods.
	 */

	virtual bool start(IOService *provider);
        virtual void stop(IOService *provider);
	virtual void free();

        /*
	 * Methods inherited from IONetworkController
	 */

	virtual IOReturn enable(IONetworkInterface *netif);
	virtual IOReturn disable(IONetworkInterface *netif);


        virtual UInt32 outputPacket(mbuf_t m_head, void *param);

        virtual void getPacketBufferConstraints(
                 IOPacketBufferConstraints *constraints) const;

	virtual IOOutputQueue *createOutputQueue();
	virtual const OSString *newVendorString() const;
        virtual const OSString *newModelString() const;

	virtual bool configureInterface(IONetworkInterface *netif);
	virtual IOReturn setMaxPacketSize(UInt32 maxSize);
	virtual IOReturn getChecksumSupport(UInt32 *checksumMask,
					    UInt32 checksumFamily,
					    bool isOutput );
        /*
	 * Methods inherited from IOEthernetController.
	 */

        virtual IOReturn getHardwareAddress( IOEthernetAddress * addr );
	virtual IOReturn getMaxPacketSize(UInt32 * maxSize) const;
	virtual IOReturn setPromiscuousMode(bool active);
	virtual IOReturn getPacketFilters(UInt32 * filters) const;

	/*
	 * our own methods
	 */
	void mx_ether_close();
	bool mx_ether_open();
	int mx_get_buf_small(uint i);
	int mx_get_buf_big(uint i);
	bool create_medium_tables(void);
	void  link_change(void);
	mbuf_t mx_m_compress(mbuf_t m_head);
	char tx_scratch[256 + ETHER_HDR_LEN + 4];
};

OSDefineMetaClassAndStructors (mx_ethernet, IOEthernetController);
#define super IOEthernetController



static void
macosx_tx_done(struct mx_instance_state *is, uint32_t mcp_index)
{
	class mx_ethernet *ethernet;
	struct mx_ether *eth;
	mbuf_t m;
	mbuf_t freelist = 0;
	uint idx;
	

	eth = is->ether;
	ethernet = (class mx_ethernet *)eth->arch.cpp_class;

	while (eth->tx.done != (int)mcp_index) {
		idx = eth->tx.done & (NUM_TX - 1);
		m = eth->tx.info[idx].m;

		/* Mark the DMA entry as free */
		eth->tx.info[idx].m = 0;
		eth->tx.done++;

		/* mbuf attached to only one slot in the packet */
		if (m) {
			/* chain packets into a free list to avoid
			   mbuf locking overhead */
			mbuf_setnextpkt(m, freelist);
			freelist = m;
		}
	}
	ethernet->net_stats->outputPackets += mbuf_freem_list(freelist);

	/* Kick the transmit routine */
	if (ethernet->stalled) {
		ethernet->stalled = 0;
		ethernet->tx_queue->service();
	}
}

static inline void
mx_tcpudp_csum(mbuf_t m, uint32_t csum)
{
	struct ether_header *eh;
	struct ip *ip;
	int iphlen;
	uint16_t c;

	eh = (struct ether_header *) (mtod(m, char *));
	ip = (struct ip *) (sizeof(*eh) + mtod(m, char *));
	iphlen = ip->ip_hl << 2;
	c = in_pseudo (ip->ip_src.s_addr, ip->ip_dst.s_addr,
		       htonl(csum + ntohs(ip->ip_len) +
			     - iphlen + ip->ip_p));
		
	c ^= 0xffff;
	if (__predict_true(c == 0)) {
		mx_set_csum_good(m);
	}

}


static void
macosx_rx_done_small(mx_instance_state_t *is, uint32_t count, uint32_t len, 
		       uint32_t csum, uint32_t flags)
{
	class mx_ethernet *ethernet;
	struct mx_ether *eth;
	mbuf_t m;
	int idx;

	eth = is->ether;
	ethernet = (class mx_ethernet *)eth->arch.cpp_class;
	idx = eth->rx_small.cnt & (NUM_RX - 1);
	eth->rx_small.cnt++;
	/* save a pointer to the received mbuf */
	m = eth->rx_small.info[idx].m;	
	/* try to replace the received mbuf */
	if (ethernet->mx_get_buf_small(idx)) {
		ethernet->net_stats->inputErrors++;
		/* drop the frame -- the old mbuf is re-cycled */
		return;
	}

	/* mcp implicitly skips 1st bytes so that
	 * packet is properly aligned */
	mbuf_adj(m, MX_MCP_ETHER_PAD);
	mbuf_pkthdr_setlen(m, len);
	mbuf_setlen(m, len);

	if (__predict_true((flags & MX_MCP_ETHER_FLAGS_CKSUM))) {
		mx_tcpudp_csum(m, csum);
	}

	ethernet->netif->inputPacket(m, 0);
	ethernet->net_stats->inputPackets++;
}

static void 
macosx_rx_done_big(mx_instance_state_t *is, uint32_t count, uint32_t len, 
		       uint32_t csum, uint32_t flags)
{
	class mx_ethernet *ethernet;
	struct mx_ether *eth;
	mx_ether_rx_buf_t *rx;
	mbuf_t m = 0; 		/* -Wunitialized */
	mbuf_t m_prev = 0;	/* -Wunitialized */
	mbuf_t m_head = 0;
	int idx, seglen;

	eth = is->ether;
	rx = &eth->rx_big;
	ethernet = (class mx_ethernet *)eth->arch.cpp_class;

	seglen = 0;
	while ((int)len > 0) {
		idx = rx->cnt & (NUM_RX - 1);
		/* save a pointer to the received mbuf */
		m = rx->info[idx].m;
		/* try to replace the received mbuf */
		if (__predict_false(ethernet->mx_get_buf_big(idx))) {
			goto drop;
		}
		/* chain multiple segments together */
		if (m_head == NULL) {
			m_head = m;
			/* mcp implicitly skips 1st bytes so that
			 * packet is properly aligned */
			mbuf_adj(m, MX_MCP_ETHER_PAD);
			mbuf_pkthdr_setlen(m, len);
			seglen = BIG_BYTES - MX_MCP_ETHER_PAD;
		} else {
			seglen = BIG_BYTES;
#if MX_DARWIN_XX <= 7
			m->m_flags &= ~M_PKTHDR;
#else
			mbuf_setflags(m, mbuf_flags(m) & ~MBUF_PKTHDR);
#endif
			mbuf_setnext(m_prev, m);
		}
		len -= seglen;
		mbuf_setlen(m, seglen);
		m_prev = m;
	}

	/* trim trailing garbage from the last mbuf in the chain.  If
	 * there is any garbage, len will be negative */
	mbuf_setlen(m, seglen + (int)len);

	if (__predict_true((flags & MX_MCP_ETHER_FLAGS_CKSUM))) {
		mx_tcpudp_csum(m_head, csum);
	}

	ethernet->netif->inputPacket(m_head, 0);
	ethernet->net_stats->inputPackets++;
	return;

drop:

	/* drop the frame -- the old mbuf(s) are re-cycled */


	/* advance to the next frame, and give the DMA addresses
	 back to the MCP */
	do {
		idx = rx->cnt & (NUM_RX - 1);
		rx->cnt++;
		if ((idx & 7) == 7) {
			mx_pio_memcpy(&rx->ring[idx - 7], &rx->shadow[idx - 7],
				    8 * sizeof (*rx->ring), 0);
			MX_STBAR();
			MX_PIO_WRITE(rx->lanai_cnt, htonl(rx->cnt));
		}
		len -= BIG_BYTES;
	} while ((int)len > 0);
	if (m_head)
		mbuf_freem(m_head);
	ethernet->net_stats->inputErrors++;
	/* tell the mcp about the buffer(s) we gave it */

}

static void
macosx_link_change(mx_instance_state_t *is)
{
	class mx_ethernet *ethernet;
	struct mx_ether *eth;
	IONetworkMedium *medium;

	eth = is->ether;
	if (eth == NULL)
		return;
	ethernet = (class mx_ethernet *)eth->arch.cpp_class;

	ethernet->link_change();
}

mbuf_t
mx_ethernet::mx_m_compress(mbuf_t m_head)
{
	mbuf_t m_src, m_dst, m_ret, md_prev;
	char *src, *dst;
	int len, slen, dlen, copy_len, dused;

	len = mbuf_pkthdr_len(m_head);
	mbuf_outbound_finalize(m_head, PF_INET, ETHER_HDR_LEN);
	m_ret = allocatePacket(len);
	if (m_ret == NULL) {
		return NULL;
	}
	m_dst = m_ret;
	m_src = m_head;
	md_prev = NULL;

	slen = dlen = dused = 0;
	dst = src = NULL;
	mbuf_pkthdr_setlen(m_ret, len);
	while (len >= 0 && m_dst && m_src)  {
		/* grab new pointers & lengths */
		if (slen == 0) {
			src = mtod(m_src, char *);
			slen = mbuf_len(m_src);
		}
		if (dlen == 0) {
			dst = mtod(m_dst, char *);
			dlen = mbuf_len(m_dst);
			dused = 0;
		}
		/* copy the min of the data remaining the the src &
		   dest. mbuf chains & the len  we need to copy */
		copy_len = (slen < dlen) ? slen : dlen;
		copy_len = (copy_len < len) ? copy_len : len;
		bcopy(src, dst, copy_len);
		/* increment pointers & decrement lengths */
		src += copy_len;
		dst += copy_len;
		slen -= copy_len;
		dlen -= copy_len;
		len -= copy_len;
		dused += copy_len;
		/* advance mbuf chain if needed */
		if (slen == 0) {
			m_src = mbuf_next(m_src);
		}
		if (dlen == 0) {
			md_prev = m_dst;
			m_dst = mbuf_next(m_dst);
		}
	}

	if (len > 0)   {
		MX_WARN(("ran out of data when copying packet for xmit\n"));
		mbuf_freem(m_ret);
		return (NULL);
	}
	/* update the len field in the last dest mbuf to
	   reflect how many bytes we're actually using */
	if (dlen != 0) {
		mbuf_setlen(m_dst, dused);
		md_prev = m_dst;
		m_dst = mbuf_next(m_dst);
	}
	/* free the rest of the m_dst chain */
	if (m_dst) {
		if (md_prev) {
			mbuf_setnext(md_prev, NULL);
		}
		mbuf_freem(m_dst);
	}
	
	return m_ret;
}

bool
mx_ethernet::start(IOService *provider)
{
	bool started;
	mbuf_t m;

#if MX_DARWIN_XX > 7
	mbuf_stats(&mbuf_info);
#endif

	driver = OSDynamicCast(mx_driver, provider);
	if (!driver)
		goto abort_with_nothing;

	is = driver->is;

	if (!is)
		goto abort_with_nothing;

	mx_workloop = driver->getWorkLoop();
	if (!mx_workloop) {
		MX_WARN(("mx%d: Unable to get workloop for mx_ether\n", is->id));
	}

	started = super::start(provider);
	if (started == false)
		return false;

	/* allocate and setup ethernet softc */
	eth = (struct mx_ether *)mx_kmalloc((sizeof *eth), MX_MZERO|MX_NOWAIT);
	if (!eth)
		goto abort_with_super;
	is->ether = eth;
	eth->is = is;

	/* save our class so that we can call C++ IOKit functions from C */
	eth->arch.cpp_class = (void *)this;
	MX_STBAR();

	is->arch.ether_tx_done = macosx_tx_done;
	is->arch.ether_rx_done_small = macosx_rx_done_small;
	is->arch.ether_rx_done_big = macosx_rx_done_big;
	is->arch.ether_link_change = macosx_link_change;

	eth->running = MX_ETH_STOPPED;

	enabled = false;

	rx_small_mcursor = IOMbufBigMemoryCursor::withSpecification(SMALL_BYTES, 1);
	rx_big_mcursor = IOMbufBigMemoryCursor::withSpecification(BIG_BYTES, 1);
	tx_mcursor = IOMbufBigMemoryCursor::withSpecification(MX_MAX_ETHER_MTU + 4,
							      MX_MCP_ETHER_MAX_SEND_FRAG); 

	if (!rx_small_mcursor || !rx_big_mcursor || !tx_mcursor) {
		IOLog("Mbuf cursor allocation failed (%p %p %p)\n",
		      rx_small_mcursor, rx_big_mcursor, tx_mcursor);
		goto abort_with_mcursor;
	}

	tx_queue = getOutputQueue();
	if (tx_queue == 0) {
		IOLog("Output queue allocation failed\n");
		goto abort_with_mcursor;
	}
	tx_queue->retain();

	if (!create_medium_tables()) {
		IOLog("Medium table creation failed\n");
		goto abort_with_tx_queue;
	}

	if (attachInterface((IONetworkInterface **) &netif, false) == false) {
		IOLog("mx%d: Failed to attach IONetworkInterface\n", is->id);
		goto abort_with_mcursor;
	}
	netif->registerService();
	netif->set_mtu(this, MX_MAX_ETHER_MTU - 14);

	return true;

abort_with_tx_queue:
	tx_queue->release();

abort_with_mcursor:
	if (tx_mcursor)
		tx_mcursor->release();
	if (rx_big_mcursor)
		rx_big_mcursor->release();
	if (rx_small_mcursor)
		rx_small_mcursor->release();
	tx_mcursor = rx_small_mcursor = rx_big_mcursor = 0;

	is->ether = 0;
	if (eth) 
		mx_kfree(eth);
	eth = 0;

abort_with_super:
	 super::stop(provider);
abort_with_nothing:
	 return false;

}

void
mx_ethernet::stop(IOService *provider)
{
	disable(netif);
	if (netif != 0) {
		IONetworkController::detachInterface((IONetworkInterface *) netif, 
					     true);
		netif->release();
		netif = 0;
	}

	if (tx_queue) 
		tx_queue->release();
	if (tx_mcursor)
		tx_mcursor->release();
	if (rx_big_mcursor)
		rx_big_mcursor->release();
	if (rx_small_mcursor)
		rx_small_mcursor->release();
	tx_mcursor = rx_small_mcursor = rx_big_mcursor = 0;

	is->arch.ether_tx_done = NULL;
	is->arch.ether_rx_done_small = NULL;
	is->arch.ether_rx_done_big = NULL;
	is->arch.ether_link_change = NULL;

	is->ether = 0;
	MX_STBAR();
	if (eth) 
		mx_kfree(eth);
	eth = 0;
	
	super::stop(provider);
}

void
mx_ethernet::free()
{
	if (medium_dict != NULL)
		medium_dict->release();

	super::free();
}

IOReturn 
mx_ethernet::getHardwareAddress(IOEthernetAddress *addrs)
{
        bcopy(is->mac_addr, addrs, sizeof(*addrs));
        return kIOReturnSuccess;
}

IOReturn 
mx_ethernet::getMaxPacketSize(UInt32 * maxSize) const
{
	/* Add 4 here to account for fcs trailer that we don't
	   use, but which the upper layers want to leave space for */
	*maxSize = MX_MAX_ETHER_MTU + 4;
	return kIOReturnSuccess;
}

IOReturn
mx_ethernet::setPromiscuousMode(bool active)
{
	mx_ether_set_promisc_common(eth, active);
	return kIOReturnSuccess;
}

IOReturn
mx_ethernet::getPacketFilters(UInt32 *filters) const
{
	*filters = (kIOPacketFilterUnicast |
		    kIOPacketFilterBroadcast |
		    kIOPacketFilterPromiscuous);
	return kIOReturnSuccess;
}

IOReturn 
mx_ethernet::setMaxPacketSize(UInt32 maxSize)
{
	int error, mtu;
	
	mtu = maxSize - ETHER_CRC_LEN ; /* strip of fcs trailer space */
	error = mx_mcpi.set_param(is->id, 
				  is->lanai.sram, 
				  "ethernet_mtu", mtu);	
	if (error == 0) {
		MX_INFO(("en%d:set mtu to %d\n", 
			 netif->getUnitNumber(), mtu - ETHER_HDR_LEN));
		return kIOReturnSuccess;
	} else {
		MX_INFO(("en%d: failed seting mtu to %d (%d)\n", 
			 netif->getUnitNumber(), mtu - ETHER_HDR_LEN, error));
		return kIOReturnIOError;
	}
}

IOReturn 
mx_ethernet::getChecksumSupport(UInt32 *checksumMask,
				UInt32 checksumFamily,
				bool isOutput )
{
	*checksumMask = kChecksumTCP|kChecksumUDP;
	return kIOReturnSuccess;
}

UInt32 
mx_ethernet::outputPacket(mbuf_t m_head, void *param)
{
	struct IOPhysicalSegment segments[MX_MCP_ETHER_MAX_SEND_FRAG];
	mcp_kreq_ether_send_t req_list[MX_MCP_ETHER_MAX_SEND_FRAG + 1];
	mcp_kreq_ether_send_t *req;
	char *header;
	mcp_kreq_ether_send_t *first_req;
	struct ip *ip;
	UInt32 demand;
	uint16_t cksum_offset, pseudo_hdr_offset;
	mbuf_t m;
	size_t mlen;

	uint avail, idx, count, nsegs;
	

	if (!enabled)
		goto abort;

	/* leave 2 slots, one to keep the ring from wrapping, and
	   one extra to account for any pullups */
	avail = NUM_TX - 2 - (eth->tx.req - eth->tx.done);

	if (avail == 0) {
		stalled = 1;
		return kIOReturnOutputStall;
	}

	for (nsegs = 1, m = m_head; m != NULL; m = mbuf_next(m)) {
		nsegs++;
	}

	if (nsegs > avail || nsegs > MX_MCP_ETHER_MAX_SEND_FRAG - 1) {
		if ((nsegs > MX_MCP_ETHER_MAX_SEND_FRAG - 1 ) &&
		    (avail >=  MX_MCP_ETHER_MAX_SEND_FRAG - 1)) {
			/* compress the chain */
			m = mx_m_compress(m_head);
			if (m == NULL) {
				MX_WARN(("Could not compress mbuf with %d segs\n",
					 nsegs));
				goto abort;
			} 
			mbuf_freem(m_head);
			m_head = m;
		} else {
			stalled = 1;
			return kIOReturnOutputStall;
		}
	}
	m = m_head;

	getChecksumDemand(m, kChecksumFamilyTCPIP, &demand);

	/* Get the DMA address of the mbufs in the chain */
	count = tx_mcursor->getPhysicalSegmentsWithCoalesce(m_head, segments, 
							    MX_MCP_ETHER_MAX_SEND_FRAG);

	if ((1 + count) > avail || count == 0) {
		MX_WARN(("bad count %d, can't be more than %d\n",
			 count, nsegs));
		goto abort;
	}

	/* Get the destination mac address for the mcp.
	   It is already in network byte order */
	req = req_list;
	header = mtod(m, char *);
	req->head.dest_high16 = *(uint16_t *)&header[0];
	req->head.dest_low32 = *(uint32_t *)&header[2];	
	req->head.flags = MX_MCP_ETHER_FLAGS_HEAD | MX_MCP_ETHER_FLAGS_VALID;
	req->head.pseudo_hdr_offset = 0;
	req->head.cksum_offset = 0;

	/* Setup checksum offloading, if needed */
	if (demand) {
		size_t ip_off = sizeof (struct ether_header);
		/* ensure ip header is contiguous in first mbuf, copy
                   it to a scratch buffer if not */
		mlen = mbuf_len(m);
		if (__predict_false(mlen < ip_off + sizeof (struct ip))) {
			mbuf_copydata(m, 0, ip_off + sizeof(struct ip),
				      tx_scratch);
			header = tx_scratch;
		} else {
			header = (char *)mbuf_data(m);
		}
		
		ip = (struct ip *) (header + sizeof (struct ether_header));
		cksum_offset = sizeof(struct ether_header) + (ip->ip_hl << 2);
		if (ip->ip_p == IPPROTO_TCP)
			pseudo_hdr_offset = cksum_offset + 
				offsetof (struct tcphdr, th_sum);
		else
			pseudo_hdr_offset = cksum_offset + 
				offsetof (struct udphdr, uh_sum);
			
		req->head.pseudo_hdr_offset = htons(pseudo_hdr_offset);
		req->head.cksum_offset = htons(cksum_offset);
		req->head.flags = MX_MCP_ETHER_FLAGS_HEAD | MX_MCP_ETHER_FLAGS_CKSUM | MX_MCP_ETHER_FLAGS_VALID;

	}


	/* walk the DMA addr / len pairs and store them where the
	   firmware can find them */

	nsegs = 0;
	while (count != 0) {
		req++;
		count--;
		req->frag.addr_low = segments[nsegs].location;
		req->frag.addr_high = 0;
		req->frag.length = htons(ntohl(segments[nsegs].length));
		req->frag.flags = MX_MCP_ETHER_FLAGS_VALID;
		nsegs++;
	}
	/* account for the header */
	nsegs++;
	/* mark it valid */
	req->frag.flags |= MX_MCP_ETHER_FLAGS_LAST;
	idx = (eth->tx.req + nsegs) & (NUM_TX - 1);
	eth->tx.info[idx].m = m_head;
	
	mx_ether_submit_tx_req(eth, req_list, nsegs);
	return kIOReturnOutputSuccess;

abort:
	freePacket(m_head);

	net_stats->outputErrors++;
	return kIOReturnOutputDropped;
}


int
mx_ethernet::mx_get_buf_small(uint idx)
{
	mbuf_t m;
	struct IOPhysicalSegment segment;
	mx_ether_rx_buf_t *rx = &eth->rx_small;
	int count;
	int retval = 0;
	
	m = allocatePacket(SMALL_BYTES);
	if (!m) {
		rx->alloc_fail++;
		retval = ENOBUFS;
		goto done;
	}
	count = rx_small_mcursor->getPhysicalSegments(m, &segment);
	if (count != 1) {
		IOLog("%s: %s -- getPhysicalSegments failed !?!?  count = %d\n", 
		      getName(), __FUNCTION__, count);
		freePacket(m);
		rx->alloc_fail++;
		retval = ENOBUFS;
		goto done;
	}
	
	rx->info[idx].m = m;
	rx->shadow[idx].addr_low = segment.location;
done:
	if ((idx & 7) == 7) {
		mx_pio_memcpy(&rx->ring[idx - 7], &rx->shadow[idx - 7],
			    8 * sizeof (*rx->ring), 0);
		MX_STBAR();
		MX_PIO_WRITE(rx->lanai_cnt, htonl(rx->cnt));
	}
	
	return retval;

}

int
mx_ethernet::mx_get_buf_big(uint idx)
{
	mx_ether_rx_buf_t *rx = &eth->rx_big;
	mbuf_t m;
	struct IOPhysicalSegment segment;
	int count;

	
	m = allocatePacket(BIG_BYTES);
	if (!m)
		goto fail;
	count = rx_big_mcursor->getPhysicalSegments(m, &segment);
	if (count != 1)
		goto fail_with_m;
	
	rx->info[idx].m = m;
	rx->shadow[idx].addr_low = segment.location;

	rx->cnt++;
	if ((idx & 7) == 7) {
		mx_pio_memcpy(&rx->ring[idx - 7], &rx->shadow[idx - 7],
			    8 * sizeof (*rx->ring), 0);
		MX_STBAR();
		MX_PIO_WRITE(rx->lanai_cnt, htonl(rx->cnt));
	}

	return 0;

fail_with_m:
	IOLog("%s: %s -- getPhysicalSegments failed !?!?  count = %d\n", 
	      getName(), __FUNCTION__, count);
	freePacket(m);

fail:
	rx->alloc_fail++;
	return ENOBUFS;


}

void
mx_ethernet::mx_ether_close()
{
	mbuf_t m;
	uint32_t dont_care;
	uint i;


	/* if buffers not alloced, give up */
	if (!eth->rx_big.shadow)
		return;

	/* if the device not running give up */
	if (eth->running != MX_ETH_RUNNING  &&
	    eth->running != MX_ETH_OPEN_FAILED)
		return;

	eth->running = MX_ETH_STOPPING;

	mx_lanai_command(is, MX_MCP_CMD_ETHERNET_DOWN,
			 0, 0, 0, &dont_care, &eth->cmd_sync);

	eth->running = 0;

	/* free recvs */
	for (i = 0; i < NUM_RX; i++) {
		eth->rx_small.shadow[i].addr_low = 0;
		m = eth->rx_small.info[i].m;
		eth->rx_small.info[i].m = 0;
		if (m)
			freePacket(m);

		eth->rx_big.shadow[i].addr_low = 0;
		m = eth->rx_big.info[i].m;
		eth->rx_big.info[i].m = 0;
		if (m)
			freePacket(m);
	}

	/* free transmits */
	for (i = 0; i < NUM_TX; i++) {
		m = eth->tx.info[i].m;
		eth->tx.info[i].m = 0;
		if (m)
			freePacket(m);
	}
	mx_ether_close_common(is);
}


bool
mx_ethernet::mx_ether_open()
{
	int error;
	uint i;
	uint32_t dont_care;

	error = mx_ether_open_common(is, 
				     netif->getMaxTransferUnit() + 14,
				     SMALL_BYTES, BIG_BYTES);
	if (error) {
		MX_WARN(("%s: mx_ether_open_common() failed, errno = %d\n",
			 getName(), error));
		goto abort_with_nothing;
	}
	
	/* allocate recvs */
	for (i = 0; i < NUM_RX; i++) {
		error = mx_get_buf_big(i);
		if (error) {
			MX_WARN(("%s: Could not alloc big recv buffer %d, errno = %d\n",
				 getName(), i, error));
			goto abort_with_open;
		}
		error = mx_get_buf_small(i);
		if (error) {
			MX_WARN(("%s: Could not alloc small recv buffer %d, errno = %d\n",
				 getName(), i, error));
			goto abort_with_open;
		}
	}

	eth->rx_small.cnt = NUM_RX;
	eth->rx_big.cnt = NUM_RX;
	MX_PIO_WRITE(eth->rx_small.lanai_cnt, htonl(eth->rx_small.cnt));
	MX_PIO_WRITE(eth->rx_big.lanai_cnt, htonl(eth->rx_big.cnt));

	/* tell the mcp about this */
	error = mx_lanai_command(is, MX_MCP_CMD_ETHERNET_UP,
				 0, 0, 0, &dont_care, &eth->cmd_sync);

	if (error) {
		MX_WARN(("%s: unable to start ethernet\n", getName()));
		goto abort_with_open;
	}
	mx_ether_start_common(is, 
			      netif->getMaxTransferUnit() + 14,
			      SMALL_BYTES, BIG_BYTES);
	eth->running = MX_ETH_RUNNING;
	return true;
	
abort_with_open:
	eth->running = MX_ETH_OPEN_FAILED;
	mx_ether_close();
		
abort_with_nothing:
	eth->running = MX_ETH_STOPPED;
	return false;
}

IOReturn
mx_ethernet::enable(IONetworkInterface *netif)
{

	if (!enabled) {
		enabled = mx_ether_open();
	}
	if (!enabled)
		return kIOReturnIOError;

	tx_queue->setCapacity(NUM_TX - 1);
	tx_queue->start();
	return kIOReturnSuccess;
}

IOReturn
mx_ethernet::disable(IONetworkInterface *netif)
{
	if (enabled) {
		tx_queue->stop();
		tx_queue->setCapacity(0);
		tx_queue->flush();
		mx_ether_close();
	}
	enabled = false;
	return kIOReturnSuccess;
}


bool
mx_ethernet::configureInterface(IONetworkInterface *netif)
{
	IONetworkData *data;

	if (super::configureInterface(netif) == false)
		return false;

	/* Get the generic network statistics structure.*/
	data = netif->getParameter(kIONetworkStatsKey);
        if (!data || !(net_stats = (IONetworkStats *)data->getBuffer())) {
                return false;
        }

        /* Get the Ethernet statistics structure. */

        data = netif->getParameter(kIOEthernetStatsKey);
        if (!data || !(ether_stats = (IOEthernetStats *)data->getBuffer())) {
                return false;
        }
        
        return true;
}

void
mx_ethernet::getPacketBufferConstraints(IOPacketBufferConstraints *constraints) const
{
	constraints->alignStart  = kIOPacketBufferAlign1;
	constraints->alignLength  = kIOPacketBufferAlign1;
}

IOOutputQueue *mx_ethernet::
createOutputQueue()
{       
	IOOutputQueue *q;

        q = IOBasicOutputQueue::withTarget(this, NUM_TX - 1);
	return q;
}

const OSString * 
mx_ethernet::newVendorString() const
{
        return OSString::withCString("Myricom");
}

const OSString * 
mx_ethernet::newModelString() const
{
	const char *model = 0;
	switch (is->pci_rev) {
	case 4:
		model = "PCIXD";
		break;
	case 5:
		model = "PCIXE";
		break;
	case 6:
		model = "PCIXF";
		break;
	default:
		model = "unknown";
		break;
	}
	return OSString::withCString(model);
}

struct mtable {
    UInt32      type;
    UInt32      speed;
};

static struct mtable medium_table[] = {
    { kIOMediumEthernetNone,    0},
    { kIOMediumEthernetAuto | kIOMediumOptionFullDuplex, 2000},
    { kIOMediumEthernetAuto | kIOMediumOptionFullDuplex, 4000},
};

bool 
mx_ethernet::create_medium_tables(void)
{
    IONetworkMedium *medium;
    size_t size;
    unsigned int i;

    size = sizeof(medium_table)/sizeof(medium_table[0]);
    medium_dict = OSDictionary::withCapacity(size);
    if (medium_dict == NULL) 
	    return false;

    for (i = 0; i < size; i++) {
	    medium = IONetworkMedium::medium(medium_table[i].type, 
					     medium_table[i].speed);
	    if (medium != NULL) {  
		    IONetworkMedium::addMedium(medium_dict, medium);
		    medium->release();
	    }
    }
    if (publishMediumDictionary(medium_dict) != true) 
	    return false;
    medium = IONetworkMedium::
	    getMediumWithType(medium_dict,
			      kIOMediumEthernetAuto|kIOMediumOptionFullDuplex);
    setCurrentMedium(medium);
#if 0
    setLinkStatus(kIONetworkLinkActive | kIONetworkLinkValid,
		  medium, 2000LL * (is->pci_rev - 3));
#endif
    link_change();
    return true;    
}

void
mx_ethernet::link_change(void)
{
	IONetworkMedium *medium;
	int i, speed, status;

	if (medium_dict == NULL)
		return;

	medium = IONetworkMedium::
		getMediumWithType(medium_dict,
				  kIOMediumEthernetAuto|kIOMediumOptionFullDuplex);
	for(speed = 0, i = 0; i < (int)is->num_ports; i++)
		if (is->link_state & (1 << i))
			speed += 2000;

	status = kIONetworkLinkValid;
	if (speed)
		status |= kIONetworkLinkActive;

//	IOLog("%s: 0x%x, %d\n", getName(), status, speed);
	setLinkStatus(status,   medium, (long long)speed);
}


/*
  This file uses MX driver indentation.

  Local Variables:
  c-file-style:"linux"
  tab-width:8
  End:
*/
